<?php
function now_dt(): string { return date('Y-m-d H:i:s'); }

function fa_norm(string $s): string {
  $s = trim($s);
  $s = str_replace(['ي','ك','ۀ','ة'], ['ی','ک','ه','ه'], $s);
  $s = preg_replace('/[\x{200c}\x{200f}]/u',' ', $s); // ZWNJ/RTL marks
  $s = preg_replace('/[\p{P}\p{S}]+/u',' ', $s);
  $s = preg_replace('/\s+/u',' ', $s);
  return trim($s);
}

function extract_keywords(string $q, int $max=4): array {
  $q = mb_strtolower(fa_norm($q));
  $parts = preg_split('/\s+/u', $q);
  $stop = ['لطفا','لطفاً','میخوام','می‌خوام','میخواهم','می‌خواهم','بده','بگو','میشه','میشود','می‌شه','می‌شود','رو','را','از','به','برای','در','و','یا','این','اون','آن','یک','یه','هم','با','که','ها','ترین','تر','کردن','کن','کنید','کنه','کنم','دارم','دارید','داره','دارند','شد','شدن','باشه','کنار','کجاست','کجاس','کجا','چی','چیه','چطور','چگونه','اطلاعات','کامل','معرفی','لیست','اول','بعد','اگر'];
  $stop = array_flip($stop);
  $out = [];
  foreach($parts as $w){
    $w = trim($w);
    if ($w==='' || isset($stop[$w])) continue;
    if (mb_strlen($w) < 3) continue;
    $out[] = $w;
  }
  usort($out, fn($a,$b)=> mb_strlen($b) <=> mb_strlen($a));
  $out = array_values(array_unique($out));
  return array_slice($out, 0, $max);
}

function boost_active(array $c): bool {
  if (empty($c['boost_enabled'])) return false;
  $start = $c['boost_start'] ?? null;
  $end   = $c['boost_end'] ?? null;
  $n = now_dt();
  if ($start && $n < $start) return false;
  if ($end && $n > $end) return false;
  return true;
}

function log_answer(PDO $pdo, string $q, string $intent, string $source, ?int $companyId=null): void {
  $ip = $_SERVER['REMOTE_ADDR'] ?? '';
  $st = $pdo->prepare("INSERT INTO chat_logs(question,intent,source,company_id,ip) VALUES(?,?,?,?,?)");
  $st->execute([$q, $intent, $source, $companyId, $ip]);
}

function match_faq(PDO $pdo, string $q): ?array {
  $rows = $pdo->query("SELECT * FROM faq WHERE is_active=1 ORDER BY priority DESC, id DESC LIMIT 300")->fetchAll();
  $ql = mb_strtolower(fa_norm($q));
  foreach($rows as $r){
    $qq = mb_strtolower(fa_norm((string)$r['question']));
    if ($qq && (mb_strpos($ql, $qq) !== false || mb_strpos($qq, $ql) !== false)) return $r;

    $kw = trim((string)($r['keywords'] ?? ''));
    if ($kw !== '') {
      $parts = preg_split('/[،,\n\r]+/u', $kw);
      foreach($parts as $p){
        $p = trim(mb_strtolower(fa_norm($p)));
        if ($p!=='' && mb_strpos($ql, $p) !== false) return $r;
      }
    }
  }
  return null;
}

function company_with_tags(PDO $pdo, array $companyRow): array {
  $st = $pdo->prepare("SELECT GROUP_CONCAT(t.name ORDER BY t.name SEPARATOR ', ') tag_names
    FROM company_tags ct JOIN tags t ON t.id=ct.tag_id WHERE ct.company_id=?");
  $st->execute([(int)$companyRow['id']]);
  $companyRow['tag_names'] = (string)($st->fetchColumn() ?: '');
  return $companyRow;
}

function detect_company_in_question(PDO $pdo, string $q): ?array {
  $st = $pdo->prepare("SELECT * FROM companies
    WHERE ? LIKE CONCAT('%', name, '%')
    ORDER BY is_vip DESC, vip_level DESC, id DESC
    LIMIT 5");
  $st->execute([fa_norm($q)]);
  $rows = $st->fetchAll();
  if (!$rows) return null;
  return company_with_tags($pdo, $rows[0]);
}

function find_companies(PDO $pdo, string $q): array {
  $like = "%$q%";
  $st = $pdo->prepare("SELECT c.*,
    (SELECT GROUP_CONCAT(t.name ORDER BY t.name SEPARATOR ', ')
      FROM company_tags ct JOIN tags t ON t.id=ct.tag_id WHERE ct.company_id=c.id) as tag_names
    FROM companies c
    WHERE c.name LIKE ? OR c.field LIKE ? OR c.booth LIKE ? OR c.website LIKE ?
    ORDER BY c.is_vip DESC, c.vip_level DESC, c.boost_enabled DESC, c.id DESC
    LIMIT 10");
  $st->execute([$like,$like,$like,$like]);
  return $st->fetchAll();
}

function detect_tag_in_question(PDO $pdo, string $q): ?string {
  $q2 = fa_norm($q);
  $st = $pdo->prepare("SELECT name FROM tags
    WHERE ? LIKE CONCAT('%', name, '%')
    ORDER BY CHAR_LENGTH(name) DESC
    LIMIT 1");
  $st->execute([$q2]);
  $t = $st->fetchColumn();
  return $t ? (string)$t : null;
}

function find_by_tag(PDO $pdo, string $tagName): array {
  $like = "%$tagName%";
  $st = $pdo->prepare("SELECT c.*,
    (SELECT GROUP_CONCAT(t.name ORDER BY t.name SEPARATOR ', ')
      FROM company_tags ct JOIN tags t ON t.id=ct.tag_id WHERE ct.company_id=c.id) as tag_names
    FROM companies c
    WHERE EXISTS (
      SELECT 1 FROM company_tags ct JOIN tags t ON t.id=ct.tag_id
      WHERE ct.company_id=c.id AND t.name LIKE ?
    )
    ORDER BY c.is_vip DESC, c.vip_level DESC, c.id DESC
    LIMIT 10");
  $st->execute([$like]);
  return $st->fetchAll();
}

function related_companies(PDO $pdo, int $companyId, int $limit=5): array {
  $limit = (int)$limit; if ($limit<=0) $limit=5;

  $sql = "SELECT c.*, r.weight
    FROM company_relations r JOIN companies c ON c.id=r.related_company_id
    WHERE r.company_id=?
    ORDER BY c.is_vip DESC, c.vip_level DESC, r.weight DESC
    LIMIT $limit";
  $st = $pdo->prepare($sql);
  $st->execute([$companyId]);
  $rows = $st->fetchAll();
  if ($rows) return $rows;

  $st = $pdo->prepare("SELECT ct.tag_id FROM company_tags ct WHERE ct.company_id=?");
  $st->execute([$companyId]);
  $tagIds = array_map(fn($r)=> (int)$r['tag_id'], $st->fetchAll());
  if (!$tagIds) return [];

  $in = implode(',', array_fill(0, count($tagIds), '?'));
  $params = array_merge($tagIds, [$companyId]);

  $sql = "SELECT c.*, COUNT(*) as common_tags
          FROM company_tags ct
          JOIN companies c ON c.id=ct.company_id
          WHERE ct.tag_id IN ($in) AND c.id<>?
          GROUP BY c.id
          ORDER BY c.is_vip DESC, c.vip_level DESC, common_tags DESC, c.id DESC
          LIMIT $limit";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  return $st->fetchAll();
}

function search_site(PDO $pdo, string $q, int $limit=3): array {
  $limit = (int)$limit; if ($limit<=0) $limit=3;

  $qNorm = fa_norm($q);
  $keys = extract_keywords($q, 4);
  if (!$keys) $keys = [$qNorm];

  $conds = [];
  $params = [];
  foreach($keys as $k){
    $conds[] = "(title LIKE ? OR content LIKE ?)";
    $like = "%$k%";
    $params[] = $like; $params[] = $like;
  }
  $whereText = implode(' OR ', $conds);

  // Filter noisy sections by default
  $filters = [];
  $qLower = mb_strtolower($qNorm);
  if (mb_strpos($qLower,'نویسنده') === false && mb_strpos($qLower,'تحریریه') === false) {
    $filters[] = "url NOT LIKE '%/author/%'";
  }
  if (mb_strpos($qLower,'دسته') === false && mb_strpos($qLower,'بایگانی') === false) {
    $filters[] = "url NOT LIKE '%/category/%'";
  }
  $whereFilter = $filters ? (' AND '.implode(' AND ', $filters)) : '';

  $sql = "SELECT title,url,LEFT(content,260) as snippet, updated_at
          FROM site_data
          WHERE ($whereText) $whereFilter
          ORDER BY updated_at DESC
          LIMIT $limit";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  return $st->fetchAll();
}

function build_company_answer(array $c, array $related=[]): array {
  $title = "اطلاعات شرکت";
  $summary = "این اطلاعات از بانک شرکت‌های کتاب نمایشگاه استخراج شده است.";
  $bullets = [];

  $name = $c['name'] ?? '';
  if ($name) $title = $name;

  if (!empty($c['field'])) $bullets[] = "حوزه فعالیت: ".$c['field'];
  $booth = trim(($c['hall'] ?? '').' '.($c['booth'] ?? ''));
  if ($booth) $bullets[] = "غرفه/سالن: ".$booth.(!empty($c['row_name']) ? (" | ردیف ".$c['row_name']) : "");
  if (!empty($c['phone'])) $bullets[] = "تلفن: ".$c['phone'];
  if (!empty($c['email'])) $bullets[] = "ایمیل: ".$c['email'];
  if (!empty($c['website'])) $bullets[] = "وبسایت: ".$c['website'];
  $loc = trim(($c['city'] ?? '').' '.($c['province'] ?? ''));
  if ($loc) $bullets[] = "موقعیت: ".$loc;

  if (!empty($c['is_vip'])) $bullets[] = "شرکت VIP (سطح ".((int)($c['vip_level'] ?? 0)).")";
  if (!empty($c['boost_enabled'])) {
    $bt = trim((string)($c['boost_text'] ?? ''));
    if ($bt) $bullets[] = "ویژه: ".$bt;
  }
  if (!empty($c['tag_names'])) $bullets[] = "Tagها: ".$c['tag_names'];

  $extra = '';
  if (!empty($c['description'])) $extra = trim((string)$c['description']);

  if ($related) {
    $extra .= ($extra? "\n\n":"") . "شرکت‌های مرتبط:";
    $i=0;
    foreach($related as $r){
      $i++;
      $extra .= "\n- ".$r['name'].(!empty($r['is_vip']) ? " (VIP)" : "");
      if ($i>=5) break;
    }
  }
  return [$title,$summary,$bullets,$extra];
}

function build_site_answer(array $hits): array {
  $title = "اطلاعات مرتبط از سایت";
  $summary = "چند صفحه مرتبط از سایت نمایشگاه پیدا شد.";
  $bullets = [];
  foreach($hits as $h){
    $t = trim((string)($h['title'] ?? ''));
    $u = trim((string)($h['url'] ?? ''));
    if ($t && $u) $bullets[] = $t." — ".$u;
    elseif ($u) $bullets[] = $u;
  }
  $extra = '';
  if ($hits) $extra = "اگر دوست داری، بگو کدوم لینک رو باز کنیم یا دقیق‌تر دنبال چی بگردم.";
  return [$title,$summary,$bullets,$extra];
}
